/* Tests cetegorical mutual exclusion with different numbers of threads.
 * Automatic checks only catch severe problems like crashes.
 */
#include <stdio.h>
#include <string.h>
#include "tests/threads/tests.h"
#include "threads/malloc.h"
#include "threads/synch.h"
#include "threads/thread.h"
#include "devices/timer.h" // Our timer_sleep()
#include "lib/random.h" //generate random numbers


#define BUS_CAPACITY 3
#define SENDER 0
#define RECEIVER 1
/* Priority level */
#define NORMAL 0
#define HIGH 1


/*
 *	initialize task with direction and priority
 *	call 
 * */
typedef struct {
	int direction; // 0=SENDER, 1=RECEIVER
	int priority;
} task_t;

typedef struct lock Lock;
typedef struct condition Condition; 


void batchScheduler(unsigned int num_tasks_send, unsigned int num_task_receive,
        unsigned int num_priority_send, unsigned int num_priority_receive);

void senderTask(void *);
void receiverTask(void *);
void senderPriorityTask(void *);
void receiverPriorityTask(void *);

void init_bus(void); /* Initialize synchronization variables */
void oneTask(task_t task);/*Task requires to use the bus and executes methods below*/
void getSlot(task_t task); /* task tries to use slot on the bus */
void transferData(task_t task); /* task processes data on the bus either sending or receiving based on the direction*/
void leaveSlot(task_t task); /* task release the slot */


Lock lock_mutex;                        // Universal lock for both condition variables.
Lock * lock_mutex_p = &lock_mutex;   

Condition waitingToGoNP[2];             // Condition Variable for Normal-Priority tasks. 
Condition * waitingToGoNP_p = waitingToGoNP; 

Condition waitingToGoHP[2];             // Condition Variable for High-Priority tasks.     
Condition * waitingToGoHP_p = waitingToGoHP; 

static int bus_usage = 0; // Current bus congenstion.                                   
static int waitersNP[2];  // Number of Normal-Priority tasks waiting.
static int waitersHP[2];  // Number of High-Priority tasks waiting.
static int currentDirection = SENDER; // Current direction of flow.


/* initializes semaphores */ 
void init_bus(void){ 
    int i; // Misc. index variable.

    random_init((unsigned int)123456789); 
    
    lock_init(lock_mutex_p);
    
    for (i = 0; i < 2; i++){
      cond_init(&waitingToGoNP_p[i]);
      cond_init(&waitingToGoHP_p[i]);
    }
}



/*
 *  Creates a memory bus sub-system  with num_tasks_send + num_priority_send
 *  sending data to the accelerator and num_task_receive + num_priority_receive tasks
 *  reading data/results from the accelerator.
 *
 *  Every task is represented by its own thread. 
 *  Task requires and gets slot on bus system (1)
 *  process data and the bus (2)
 *  Leave the bus (3).
 */

void batchScheduler(unsigned int num_tasks_send, unsigned int num_task_receive,
        unsigned int num_priority_send, unsigned int num_priority_receive)
{
    unsigned int i;
    /* create sender threads */
    for(i = 0; i < num_tasks_send; i++)
        thread_create("sender_task", 1, senderTask, NULL);

    /* create receiver threads */
    for(i = 0; i < num_task_receive; i++)
        thread_create("receiver_task", 1, receiverTask, NULL);

    /* create high priority sender threads */
    for(i = 0; i < num_priority_send; i++)
       thread_create("prio_sender_task", 1, senderPriorityTask, NULL);

    /* create high priority receiver threads */
    for(i = 0; i < num_priority_receive; i++)
       thread_create("prio_receiver_task", 1, receiverPriorityTask, NULL);
}

/* Normal task,  sending data to the accelerator */
void senderTask(void *aux UNUSED){
        task_t task = {SENDER, NORMAL};
        oneTask(task);
}

/* High priority task, sending data to the accelerator */
void senderPriorityTask(void *aux UNUSED){
        task_t task = {SENDER, HIGH};
        oneTask(task);
}

/* Normal task, reading data from the accelerator */
void receiverTask(void *aux UNUSED){
        task_t task = {RECEIVER, NORMAL};
        oneTask(task);
}

/* High priority task, reading data from the accelerator */
void receiverPriorityTask(void *aux UNUSED){
        task_t task = {RECEIVER, HIGH};
        oneTask(task);
}

/* abstract task execution*/
void oneTask(task_t task) {
    getSlot(task);
    transferData(task);
    leaveSlot(task);
}




/* task tries to get slot on the bus subsystem */
void getSlot(task_t task) 
{
    lock_acquire(lock_mutex_p);
    
    while((bus_usage == BUS_CAPACITY) || \
          (bus_usage > 0 && task.direction != currentDirection) || \
          (waitersHP[currentDirection] > 0 && task.priority == NORMAL)) // If the current thread task is normal-priority, but high-priority tasks are waiting, block and give way.
    {
    
      if (task.priority == HIGH){
        waitersHP[task.direction]++;
      }else{
        waitersNP[task.direction]++;
      }
      
      if (task.priority == HIGH){
        cond_wait(&waitingToGoHP_p[task.direction], lock_mutex_p); // bus_usage waiting at a barrier on either side.
      }else{
        cond_wait(&waitingToGoNP_p[task.direction], lock_mutex_p);
      }
      
      if (task.priority == HIGH){
        waitersHP[task.direction]--;
      }else{
        waitersNP[task.direction]--;
      }
    }      
    
    bus_usage++;

    currentDirection = task.direction;

    lock_release(lock_mutex_p);
}
    
/* task processes data on the bus send/receive */
void transferData(task_t task) 
{
    int64_t randomTime = (int64_t)(random_ulong() % 200); // Generate random int with range 0..n. Upper bound defined by modulo.
    timer_sleep(randomTime); /* Sleep a random amount of time */ 
    
}

/* task releases the slot */
void leaveSlot(task_t task) 
{
    lock_acquire(lock_mutex_p);

    bus_usage--; 
    
    if (waitersHP[currentDirection] > 0){
      cond_signal(&waitingToGoHP_p[currentDirection], lock_mutex_p); // If there are high-priority tasks waiting to go in same direction.
    }
    else if (waitersNP[currentDirection] > 0){
      cond_signal(&waitingToGoNP_p[currentDirection], lock_mutex_p); // If there are normal-priority tasks waiting to go in same direction.
    }
    else if (bus_usage == 0){
      cond_broadcast(&waitingToGoNP_p[1-currentDirection], lock_mutex_p); // Send signal to multiple tasks waiting to go in other direction.
      cond_broadcast(&waitingToGoHP_p[1-currentDirection], lock_mutex_p); // Send signal to multiple tasks waiting to go in other direction.
      currentDirection = 1-currentDirection;
    }
    
    
    lock_release(lock_mutex_p);
}
