#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <readline/readline.h>
#include <readline/history.h>
#include "parse.h"
#include <fcntl.h>
#include <sys/stat.h>

#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <errno.h>

#define DEBUG 0

#define TRUE 1
#define FALSE 0

void RunCommand(int, Command *);
void DebugPrintCommand(int, Command *);
void PrintPgm(Pgm *);
void stripwhite(char *);

pid_t cpidBackgroundBuffer[1000];

int main(void)
{
  Command cmd;
  int parse_result;

  /* MAIN SHELL LOOP */
  while (TRUE)
  {
    signal(SIGINT, SIG_IGN); // Ignore SIGINT in "parent".
    char *line;
    line = readline("0w0~$ ");

    if (!line){
      printf("\n"); 
      break;
    }  /* If EOF encountered, exit shell. */
    stripwhite(line);   /* Remove leading and trailing whitespace from the line. */
    if (*line){         /* If stripped line not blank. */
      add_history(line);
      parse_result = parse(line, &cmd);
      
      
      if(!strcmp(cmd.pgm->pgmlist[0], "exit")) exit(0);
      if(!strcmp(cmd.pgm->pgmlist[0], "cd")) 
        chdir(cmd.pgm->pgmlist[1]);
      else
        RunCommand(parse_result, &cmd); // Our thing!
    }

    free(line); /* Clear memory */
  }
  return 0;
}

void sigHandler(int sig){
  
  switch(sig){
    case SIGCHLD:
      if(DEBUG)fprintf(stderr, "SIGCHLD detected!\n"); // Print to stderr, as stdout has likely been redirected.
      int i = 0;
      while (cpidBackgroundBuffer[i++] != 0){
        waitpid(cpidBackgroundBuffer[i], NULL, WNOHANG);
        
      }
      
      break;
      
    case SIGINT:
      if(DEBUG)fprintf(stderr, "SIGINT detected!\n");
      exit(EXIT_SUCCESS);
      break;
      
    default:
    break;
  }

}


char **getArgs(int depth, Pgm *p){
  // eg. depth == 0 -> grab programs at this depth. No traversal.
  //     depth == 2 -> traverse twice and grab program.
  
  while(depth > 0){
    p = p->next;
    depth--;
  }
  
  return p->pgmlist;
}

/* Count number of programs separated by pipe */
int llLength(Pgm *p){
  int i =0;
  
  while(p != NULL){ // traverse linked-list and count.
    i++;
    p = p->next;
  }
  
  if(DEBUG) printf("DEBUG: pgmLength: %d\n", i);
  
  return i;
}
  
  

  
///#######################################################################
/* Execute the given command(s).
 * Note: The function currently only prints the command(s).
 * 
 * TODO: 
 * 1. Implement this function so that it executes the given command(s).
 * 2. Remove the debug printing before the final submission.
 */
 ///#######################################################################
void RunCommand(int parse_result, Command *cmd)
{
  if(DEBUG)DebugPrintCommand(parse_result, cmd); /* DELETE LATER */
  if(DEBUG)printf("Base process: %d\n", getpid());
  
  if (cmd->background == TRUE){
    signal(SIGCHLD, sigHandler);
  }  

  pid_t cpid;  // PID of processes after fork.
  int noOfChildren = llLength(cmd->pgm);
  pid_t cpidBuffer[noOfChildren]; /* List of child cpid's when spawning multiple */
  
  
  
  int fdRedirOut;  
  int fdRedirIn;
  
  /* Set up all pipes that we need */
  int pipefd[noOfChildren][2];
  int n;
  int pipeRet;
  for(n = 0; n < noOfChildren-1; n++){
    if(DEBUG)printf("Creating pipe %d\n", n);
    if (pipe(pipefd[n]) == -1){
      printf("%s\n", strerror(errno));
    }
  }
  
  
  /* BEFORE FORK */
  /* Spawn child processes */
  int i;
  for(i=0; i<noOfChildren; i++){
    cpid = fork();
    if (cpid == -1){ // fork() ERROR.
      printf("%s\n", strerror(errno));
      exit(EXIT_FAILURE);
    }
    if (cpid == 0){ // CHILD_i
      if (cmd->background == FALSE)
        signal(SIGINT, sigHandler);      
      
      if(DEBUG)printf("I'm a child: %d PID: %d\n", i, getpid());
    

      /**** Redirection ****/
      /* FIRST bin in pipe sequence */
      if (i == 0){ 
        if (cmd->rstdin){
          if(DEBUG)fprintf(stderr, "Redirecting STANDARD INPUT to file: %s\n", cmd->rstdin);
          fdRedirIn = open(cmd->rstdin, O_RDONLY); // Open file for reading.
          
          if (fdRedirIn == -1){ // File open ERROR.
            printf("%s\n", strerror(errno));
            exit(EXIT_FAILURE);
          }
          else{
            close(STDIN_FILENO);
            if ( dup2(fdRedirIn, STDIN_FILENO) == -1 ){ // Redirect stdin to file.
              printf("%s\n", strerror(errno));
              exit(EXIT_FAILURE);
            }
          }
        }
        
        /* Set up pipe connection */
        if (noOfChildren > 1){
          if(DEBUG)fprintf(stderr, "Setting up pipe 0\n");
          

          
          close(STDOUT_FILENO);
          dup2(pipefd[0][1], STDOUT_FILENO); // Redirect stdout to pipe's write-end.
          for(n = 0; n < noOfChildren-1; n++){ // Close all remaining pipes.
            close(pipefd[n][0]);
            close(pipefd[n][1]);
          }
        }
        
      }
      
      /* LAST bin in pipe sequence */
      if (i == noOfChildren-1){ 
      
        /* Redirect stdout to file (in child). */
        if (cmd->rstdout){
          if(DEBUG)printf("Redirecting STANDARD OUTPUT to file: %s\n", cmd->rstdout);
          fdRedirOut = open(cmd->rstdout, O_CREAT|O_WRONLY|O_TRUNC, S_IRUSR|S_IWUSR); // Open file for writing.
          
          if (fdRedirOut == -1){ // File open ERROR.
            printf("%s\n", strerror(errno));
            exit(EXIT_FAILURE);
          }
          else{
            close(STDOUT_FILENO);
            if ( dup2(fdRedirOut, STDOUT_FILENO) == -1 ){
              printf("%s\n", strerror(errno));
              exit(EXIT_FAILURE);
            }
          }
        }
        
        /* Set up pipe connection */
        if (noOfChildren > 1){
          if(DEBUG)fprintf(stderr, "Setting up pipe n-1\n");
         
          close(STDIN_FILENO);
          dup2(pipefd[i-1][0], STDIN_FILENO); // Redirect stdin to pipe's read-end.
          for(n = 0; n < noOfChildren-1; n++){ // Close all remaining pipes.
            close(pipefd[n][0]);
            close(pipefd[n][1]);
          }
          
          
        }
        
      }
      
      if (i > 0 && i < noOfChildren-1){ // Neither first nor last in pipe sequence.
        // eg. i = 1 -> connect to pipe0 input, pipe1 output.
        if(DEBUG)printf("Setting up pipe %d\n", i);
        
        //close(pipefd[i-1][1]); // Close write-end on input pipe.
        //close(pipefd[i][0]); // Close read-end on output pipe.
        
        dup2(pipefd[i-1][0], STDIN_FILENO); // Redirect stdin to pipe's read-end.
        dup2(pipefd[i][1], STDOUT_FILENO);  // Redirect stdout to pipe's write-end.
        
        for(n = 0; n < noOfChildren-1; n++){ // Close all remaining pipes.
            close(pipefd[n][0]);
            close(pipefd[n][1]);
        }
      }
      
      
      
      
      /* Read string of bin and execute */
      char **execArgs = getArgs(noOfChildren-i-1, cmd->pgm);
      if(DEBUG) fprintf(stderr, "Child %d: getArg: %s\n", i, execArgs[0] ); 
      if(DEBUG) fprintf(stderr, "#### EXECUTING %s ####\n", execArgs[0]); 
      int execRet = execvp(execArgs[0], execArgs); 
      if (execRet != 0){
          printf("ERROR: [%s]: %s\n", execArgs[0], strerror(errno));
      }
      
      exit(EXIT_SUCCESS); // Remember to terminate children!
    }
    if (cpid > 0){ //PARENT
      if(DEBUG)printf("I'm the parent: %d PID: %d\n", i, getpid());
      cpidBuffer[i] = cpid;
    }
  }
  
  /* Cleanup */
  if (cpid > 0){ // PARENT 
  
    for(n = 0; n < noOfChildren-1; n++){
      if(DEBUG)printf("Closing pipe %d in parent\n", n);
      close(pipefd[n][0]);
      close(pipefd[n][1]);
    }
    
  
    if (cmd->background == FALSE){
      for(i=0; i < noOfChildren; i++){ // Wait and collect ALL children.
          waitpid(cpidBuffer[i], NULL, 0);
      }
      if(DEBUG)printf("All children TERMINATED\n");
    }
    else{
      for(i = 0; i < noOfChildren+1; i++){ 
        if (i < noOfChildren)
          cpidBackgroundBuffer[i] = cpidBuffer[i];
        else
          cpidBackgroundBuffer[i] = 0;
      }
    }
    
  
    
    
    
  }
}
///#######################################################################



/* 
 * Print a Command structure as returned by parse on stdout. 
 * 
 * Helper function, no need to change. Might be useful to study as inpsiration.
 */
void DebugPrintCommand(int parse_result, Command *cmd)
{
  if (parse_result != 1) {
    printf("Parse ERROR\n");
    return;
  }
  printf("------------------------------\n");
  printf("Parse OK\n");
  printf("stdin:      %s\n", cmd->rstdin ? cmd->rstdin : "<none>");
  printf("stdout:     %s\n", cmd->rstdout ? cmd->rstdout : "<none>");
  printf("background: %s\n", cmd->background ? "true" : "false");
  printf("Pgms:\n");
  PrintPgm(cmd->pgm);
  printf("------------------------------\n");
}


/* Print a (linked) list of Pgm:s.
 * Helper function, no need to change. Might be useful to study as inpsiration.
 */
void PrintPgm(Pgm *p)
{
  if (p == NULL)
  {
    return;
  }
  else
  {
    char **pl = p->pgmlist;

    /* The list is in reversed order so print
     * it reversed to get right
     */
    PrintPgm(p->next);
    printf("            * [ ");
    while (*pl)
    {
      printf("%s ", *pl++);
    }
    printf("]\n");
  }
}


/* Strip whitespace from the start and end of a string. 
 *
 * Helper function, no need to change.
 */
void stripwhite(char *string)
{
  register int i = 0;

  while (isspace(string[i]))
  {
    i++;
  }

  if (i)
  {
    strcpy(string, string + i);
  }

  i = strlen(string) - 1;
  while (i > 0 && isspace(string[i]))
  {
    i--;
  }

  string[++i] = '\0';
}
