--------------------------------------------------------------------------------
-- Company: Chalmers University of Technology
-- Engineer: Dmitry Knyaginin
-- 
-- Create Date:    17:51:45 11/15/2012 
-- Design Name: 
-- Module Name:    top - rtl 
-- Project Name: 
-- Target Devices: 
-- Tool versions: 
-- Description: This is the top level design, and so the ports assigned to
--              pins have suffix "_pad". The design implements an indicator
--              with variable frequency of blinking. The frequency can be
--              increased and decreased with two switches or buttons on the
--              development board. Since the switch / button contacts are
--              imperfect, the respective signals on the FPGA pins are
--              bouncing for a short period after switching. Such bouncing
--              can be eliminated by filters (debouncers).
--
-- Dependencies: 
--
-- Revision: 
-- Revision 0.01 - File Created
-- Additional Comments: 
--
--------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

-- Uncomment the following library declaration if using
-- arithmetic functions with Signed or Unsigned values
use IEEE.NUMERIC_STD.ALL;

-- Uncomment the following library declaration if instantiating
-- any Xilinx primitives in this code.
library UNISIM;
use UNISIM.VComponents.all;

entity TOP is
  port (
    clk_pad : in std_logic;
    rst_pad : in std_logic;
    inc_pad : in std_logic;
    dec_pad : in std_logic;
    fil_pad : in std_logic;
    led_pad : out std_logic_vector(15 downto 0);
    an      : out std_logic_vector(7 downto 0);
    ssg     : out std_logic_vector(7 downto 0)
    );
end TOP;

architecture rtl of TOP is

-- Filter (debouncer)
component FILTER
  port (
    clk  : in std_logic;
    rst  : in std_logic;
    sin  : in std_logic;
    sout : out std_logic
    );
end component FILTER;

-- the number of LEDs
constant LED_WIDTH : integer := 16;

-- width of the free-running counter
constant CNT_WIDTH   : integer := 27;

-- most-significant bit of Cnt-Inc (wrap around it)
constant CNT_INC_WRAP : integer := 8; 

-- default value of Cnt-Inc
constant CNT_INC_DEF : unsigned(CNT_WIDTH - 1 downto 0)
  := to_unsigned(1, CNT_WIDTH);

-- divide the system clock to capture slow events such as button press/release
constant CLK_SLOW_CNT_WIDTH  : integer := 16;
signal clk_slow_cnt : unsigned(CLK_SLOW_CNT_WIDTH - 1 downto 0);
signal clk_slow : std_logic;

-- Signals just before IO buffers
signal led : std_logic_vector(LED_WIDTH - 1 downto 0);
signal clk, clk_unbuf, rst, inc, dec, fil : std_logic;

-- free-running counter and Cnt-Inc
signal cnt, cnt_inc : unsigned(CNT_WIDTH - 1 downto 0);

-- for refreshing the 7-segment indicators
constant SSG_CNT_WIDTH : integer := 19;
signal ssg_cnt : unsigned(SSG_CNT_WIDTH - 1 downto 0);

-- for internal signals and edge detectors
signal inc_int, inc_int_reg, inc_int_le : std_logic;
signal dec_int, dec_int_reg, dec_int_le : std_logic;
signal inc_fil, dec_fil : std_logic;

-- ILA 
--component ila_0
--  port (
--    clk : in std_logic;
--    probe0 : in std_logic;
--    probe1 : in std_logic;
--    probe2 : in std_logic;
--    probe3 : in std_logic;
--    probe4 : in std_logic;
--    probe5 : in std_logic_vector(CNT_INC_WRAP downto 0);
--    probe6 : in std_logic_vector(CNT_WIDTH - 1 downto 0)
--    );
--end component;
--
--component ila_1
--  port (
--    clk : in std_logic;
--    probe0 : in std_logic;
--    probe1 : in std_logic;
--    probe2 : in std_logic;
--    probe3 : in std_logic;
--    probe4 : in std_logic;
--    probe5 : in std_logic;
--    probe6 : in std_logic_vector(CNT_INC_WRAP downto 0);
--    probe7 : in std_logic_vector(CNT_WIDTH - 1 downto 0)
--    );
--end component;

-- 2x slower clock (for ila only)
signal clk_div2 : std_logic;

-- 2x slower clk_slow (for ila_slow only)
signal clk_slow_div2 : std_logic;

-- for Set Up Debug only
signal cnt_stdlv     : std_logic_vector(CNT_WIDTH - 1 downto 0);
signal cnt_inc_stdlv : std_logic_vector(CNT_WIDTH - 1 downto 0);

-- attribute mark_debug : string;
-- attribute mark_debug of clk : signal is "true";
-- attribute mark_debug of clk_div2 : signal is "true";
-- attribute mark_debug of clk_slow : signal is "true";
-- attribute mark_debug of clk_slow_div2 : signal is "true";
-- attribute mark_debug of fil : signal is "true";
-- attribute mark_debug of inc : signal is "true";
-- attribute mark_debug of dec : signal is "true";
-- attribute mark_debug of inc_int : signal is "true";
-- attribute mark_debug of dec_int : signal is "true";
-- attribute mark_debug of inc_int_le : signal is "true";
-- attribute mark_debug of dec_int_le : signal is "true";
-- attribute mark_debug of cnt_stdlv : signal is "true";
-- attribute mark_debug of cnt_inc_stdlv : signal is "true";


begin

  -- Need to instantiate input and output buffers because ILA
  -- cannot connect to pads directly
  clk_ibufg : IBUFG 
    generic map (
      IOSTANDARD => "DEFAULT"
      )
    port map (
      I => clk_pad,
      O => clk_unbuf      
      );
  
  clk_bufg : BUFG
    port map (
      I => clk_unbuf,
      O => clk
      );
  
  rst_buf : IBUF
    port map (
      I => rst_pad,
      O => rst
      );
  
  inc_buf : IBUF
    port map (
      I => inc_pad,
      O => inc
      );
  
  dec_buf : IBUF
    port map (
      I => dec_pad,
      O => dec
      );
  
  fil_buf : IBUF
    port map (
      I => fil_pad,
      O => fil
      );
        
  led_buf : for i in LED_WIDTH -1 downto 0 generate
    led_buf_i : OBUF
      port map (
        I => led(i),
        O => led_pad(i)
        );
  end generate led_buf;  

  inc_filter : FILTER
    port map (
      clk  => clk,
      rst  => rst, 
      sin  => inc, 
      sout => inc_fil
      );
  
  dec_filter : FILTER
    port map (
      clk  => clk,
      rst  => rst, 
      sin  => dec, 
      sout => dec_fil
      );
  
  --ila : ila_0
  --  port map (
  --    clk => clk,
  --    probe0 => clk_div2,
  --    probe1 => inc,
  --    probe2 => inc_int_le,
  --    probe3 => dec,
  --    probe4 => dec_int_le,
  --    probe5 => std_logic_vector(cnt_inc(CNT_INC_WRAP downto 0)),
  --    probe6 => std_logic_vector(cnt)
  --    );
  --
  --ila_slow : ila_1
  --  port map (
  --    clk => clk_slow,
  --    probe0 => clk_slow_div2,
  --    probe1 => fil,
  --    probe2 => inc,
  --    probe3 => inc_int,
  --    probe4 => dec,    
  --    probe5 => dec_int,
  --    probe6 => std_logic_vector(cnt_inc(CNT_INC_WRAP downto 0)),
  --    probe7 => std_logic_vector(cnt)
  --    );  

  -- for Set Up Debug only
  cnt_stdlv <= std_logic_vector(cnt);
  cnt_inc_stdlv <= std_logic_vector(cnt_inc);
  
  -- Route signals from the inc and dec switches (filter on/off)
  inc_int <= inc_fil when fil = '1' else inc;
  dec_int <= dec_fil when fil = '1' else dec;

  -- 7-segment indicators
  -- order: DP CG ... CA (see Nexys4 documentation)
  -- active Low

  -- refresh the blinking indicator and the filter on/off indicator in turns.
  -- multiplex the value
  ssg <= ( '1' & (not cnt(cnt'high)) & "111111" ) when 
    ssg_cnt(ssg_cnt'high) = '0' else ( '1' & "0001110" ) when
    fil = '1' else x"FF";

  -- multiplex the indicator
  an  <= (not ssg_cnt(ssg_cnt'high)) & "11" & ssg_cnt(ssg_cnt'high) & x"F";
          
  -- Display Cnt-Inc on the LEDs
  led(LED_WIDTH - 1 downto 0) <= cnt_inc_stdlv(LED_WIDTH - 1 downto 0);

  -- Leading edge detectors
  inc_int_le <= inc_int and not inc_int_reg;
  dec_int_le <= dec_int and not dec_int_reg;


  -- 7-segment control
  ssg_ctrl : process(rst, clk)
  begin
    if (rst = '1') then
      ssg_cnt <= (others => '0');
    
    elsif rising_edge(clk) then
      ssg_cnt <= ssg_cnt + 1;
      
    end if;
    
  end process ssg_ctrl;

  
  -- The free-running counter control
  cnt_ctrl : process(rst, clk) 
  begin
    if (rst = '1') then 
      cnt <= (others => '0');
    
    elsif rising_edge(clk) then
      cnt <= cnt + cnt_inc;
                  
    end if;
  
  end process cnt_ctrl;

  
  -- The Cnt-Inc control
  cnt_inc_ctrl : process(rst, clk)
  begin
    if (rst = '1') then
      cnt_inc <= CNT_INC_DEF;
      inc_int_reg <= '0';
      dec_int_reg <= '0';
  
    elsif rising_edge(clk) then
      inc_int_reg <= inc_int;
      dec_int_reg <= dec_int;
      
      -- Increase the increment value at the leading edge of the
      -- signal from the increase switch
      if (inc_int_le = '1') then
      
        -- wrap around
        if (cnt_inc(CNT_INC_WRAP) = '1') then
          cnt_inc <= (others => '0');
          cnt_inc(0) <= '1';
          
        else
          cnt_inc <= cnt_inc sll 1;
          
        end if;
      
      -- Decrease the increment value at the leading edge of the 
      -- signal from the descrease switch
      elsif (dec_int_le = '1') then
      
        -- wrap around
        if (cnt_inc(0) = '1') then
          cnt_inc <= (others => '0');
          cnt_inc(CNT_INC_WRAP) <= '1';
          
        else
          cnt_inc <= cnt_inc srl 1;
          
        end if;
              
      end if;
                  
    end if; 
  
  end process cnt_inc_ctrl;

  -- Get a clock 2x slower than clk. We need it only as a reference
  -- signal in ILA (for ease of reading waveforms)
  clk_div2_gen : process(rst, clk)
  begin
    if (rst = '1') then
      clk_div2 <= '0';
    	
    elsif rising_edge(clk) then
      clk_div2 <= not clk_div2;
    	
    end if;
  	
  end process clk_div2_gen;

  
  clk_slow_gen : process(rst, clk)
  begin
    if (rst = '1') then
      clk_slow_cnt <= (others => '0');
    	
    elsif rising_edge(clk) then
      clk_slow_cnt <= clk_slow_cnt + 1;
    	
    end if;
  	
  end process;

  clk_slow <= clk_slow_cnt(clk_slow_cnt'high - 1);
  clk_slow_div2 <= clk_slow_cnt(clk_slow_cnt'high);

end rtl;
