--************************************************************************************************************--
--! @file file_name.vhd
--************************************************************************************************************--
--========================================== Libraries ==========================================
--! Standard library
library ieee;       
--! Standard packages              
use ieee.std_logic_1164.all;      
use ieee.numeric_std.all;

use work.pkg_tb_alu.all;

--========================================== Entity ==========================================
entity ALU_TB is
generic(width : natural := 32);
end entity ALU_TB;

architecture BEHAVIORAL of ALU_TB is


-- Signal declarations
signal   clk     : std_logic:='0';
signal   Reset_n : std_logic:='0';
signal   op_A       : std_logic_vector(31 downto 0);
signal   op_B       : std_logic_vector(31 downto 0);
signal   op_Op      : std_logic_vector(3 downto 0);
signal   DUT_Outs    : std_logic_vector(31 downto 0);

signal timestep : natural := 0;
signal start_testing : std_logic := '0';

signal AMem      : Operand_array := (others => (others => '0'));
signal BMem      : Operand_array := (others => (others => '0'));
signal OpMem     : OpCode_array := (others => (others => '0'));
signal OutputMem : Operand_array := (others => (others => '0'));

-- expected output
signal exp_op_out : std_logic_vector(31 downto 0);
signal t_clk : time := 20 ns;

--! Component declaration for ALU
component ALU_SKL is
generic (width : natural);
port(
   clk   : in std_logic;
   Reset_n : in std_logic;
   A     : in std_logic_vector(31 downto 0);
   B     : in std_logic_vector(31 downto 0);
   Op       : in std_logic_vector(3 downto 0);
   Outs  : out std_logic_vector(31 downto 0));
end component;
begin

   --! Port map declaration for ALU of choice...
   DUT : ALU_SKL
      generic map (width => width)
      port map (Clk     => Clk,
                Reset_n => Reset_n,
                A       => op_A,
                B       => op_B,
                Op      => op_Op,
                Outs    => DUT_Outs);

   tb : process(clk)
   begin
      if rising_edge(clk) then
         timestep <= timestep + 1;
         
         if timestep = 2 then
            Reset_n <= '1';
            start_testing <= '1';
         end if;
      end if;
   end process;
   
   clk <= not clk after (t_clk/2);
   
   AMem        <= loadOperand(string'("A.tv"));
   BMem        <= loadOperand(string'("B.tv"));
   OpMem       <= loadOpCode( string'("Op.tv"));
   OutputMem   <= loadOperand(string'("Results.tv"));

   input_vectors:
   process 
      variable i : integer :=0; 
   begin
      wait until Reset_n='1';
      wait for t_clk/10;
      while_loop: while i <= Size loop 
            -- set inputs
            op_A    <= AMem(i);
            op_B    <= BMem(i);
            op_Op    <= OpMem(i);
            --exp_op_out <= OutputMem(i);
         
            --generate expected output
            --case opcode is
            --when "00" => OutputMem <= op_a + op_b;
            --when "01" => OutputMem <= op_a + op_b;
            --when "10" => OutputMem <= op_a nand op_b;
            --when "11" => OutputMem <= not op_b; 
            --when others => assert false;
            --end case ;


            wait for t_clk;  -- wait 1 clk period.
            i := i + 1; -- Goto next test vector
      end loop while_loop; 
   end process;
   
   output_vector:
   process 
      variable i : integer :=0; 
   begin

      wait until Reset_n='1';
      wait for 2*t_clk + t_clk/10;   -- wait 1 clock period. Output check should occur 1 clk after input set.
      
      while_loop: while i <= Size loop 
            -- set operands and opcode

            exp_op_out <= OutputMem(i);

            wait for 5 ns;
            assert(DUT_Outs = exp_op_out ) 
            --assert(false)
            report "Test failed at " & lf & 
            "op_A :" & integer'image(to_integer(signed(AMem(i)))) & lf & 
            "op_B :" & integer'image(to_integer(signed(BMem(i)))) & lf & 
            "op_Op :" & integer'image(to_integer(unsigned(OpMem(i)))) & lf &
            "EXPECTED :" & integer'image(to_integer(unsigned(exp_op_out))) & 
            "       GOT : " & integer'image(to_integer(unsigned(DUT_Outs)))
            severity failure ; --assert if unexpected output
            
            wait for t_clk - 5 ns; 
            i := i + 1; -- Goto next test vector
      end loop while_loop; 
   end process;
   

end architecture BEHAVIORAL;
