--****************************************************************************************************************--
--! alu_wRCA
--! ALU with ripple-carry adder (RCA)
--
--! 8-bit ALU implemented with RCA (Ripple Carry Adder).
--! Capable of ADD, SUB, AND, NOT operations, selected using a 4:1 mux.
--! Sets Carry, Eq, NotEq, isOutZero FLAGS.
--
--! ORIGINAL AUTHORS
--! John Croft
--! Binh Nguyen

--! Modified by John
--****************************************************************************************************************--

----------------------------------------------------- LIBRARIES ----------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use work.alu_pkg.all;
----------------------------------------------------- ENTITY -------------------------------------------------------
entity alu_wRCA_ALT is
port(
		ALU_inA, ALU_inB			:	in std_logic_vector(7 downto 0); 	--8 Bit input value.
		ALU_out						: 	out std_logic_vector(7 downto 0); 	--8 Bit output value.
		Carry, NotEq, Eq, isOutZero	: 	out std_logic;						--1 Bit on each of these signals. Eventually stored in FReg
		operation					: 	in std_logic_vector(1 downto 0) 	-- Mux select. OP: 00 ADD, 01 SUB, 10 AND, 11 NOT
		);
end alu_wRCA_ALT;
----------------------------------------------------- ARCHITECTURE ------------------------------------------------- 
architecture struct of alu_wRCA is
----------------------------------------------------- SIGNALS ------------------------------------------------------
	signal adder_out 		: std_logic_vector(7 downto 0); -- Signal connecting output from adder (ie. both ADD and SUB) to mux input.
	signal adder_in_B		: std_logic_vector(7 downto 0); -- B input to adder. Takes the value of 'ALU_inB' or its complement, 'ALU_inB_komp' when in ADD or SUB mode, respectively.
begin
	-- SUB B input inverter
	G_adder_in_B: for i in 0 to 7 generate
		adder_in_B(i) <= ALU_inB(i) XOR operation(0); -- ALU_inB complemented if in SUB mode, ie. OP=="01".
	end generate;
	
	-- RCA
	RCA8 : RCA
		generic map(N => 8)
		port map(	Cin => operation(0), 
					A => ALU_inA, 
					B => adder_in_B, 
					S => adder_out, 
					Cout => Carry);
					
	-- CMP 
	CMP0 : CMP 
		PORT MAP(	A => ALU_inA, 
					B => ALU_inB, 
					Eq => Eq, 
					NotEq => NotEq );
					
	-- isOutZero
	isOutZero <= '1' when ALU_out = x"00" else '0';
	
	-- ALU op select
	with operation select
		ALU_out	<=	adder_out 				when "00" | "01", 
					(ALU_inA AND ALU_inB) 	when "10",
					(NOT ALU_inA) 			when others;
end struct;