--****************************************************************************************************************--
--! @ EDA322Testbench
--! @ This testbench reads in expected values from external files and compares them to actual values written by the instantiated processor to the 7-seg displays.
--! @ If there are no errors, the testbench is passed after running approx. 2us in simulation.
--
--! Authors: 
--! John William Croft 
--! Alfred Kulldorff
--****************************************************************************************************************--

library IEEE;
use IEEE.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_unsigned.all;
use std.textio.all;


entity EDA322Testbench is
end EDA322Testbench;

architecture behv of EDA322Testbench is

component eda322_processor is
	Port ( 	externalIn 			: in STD_LOGIC_VECTOR (7 downto 0); -- “extIn” in Figure 1
			CLK 				: in STD_LOGIC;
			master_load_enable	: in STD_LOGIC;
			ARESETN 			: in STD_LOGIC;
			pc2seg 				: out STD_LOGIC_VECTOR (7 downto 0); -- PC
			instr2seg 			: out STD_LOGIC_VECTOR (11 downto 0); -- Instruction register
			Addr2seg 			: out STD_LOGIC_VECTOR (7 downto 0); -- Address register
			dMemOut2seg 		: out STD_LOGIC_VECTOR (7 downto 0); -- Data memory output
			aluOut2seg 			: out STD_LOGIC_VECTOR (7 downto 0); -- ALU output
			acc2seg 			: out STD_LOGIC_VECTOR (7 downto 0); -- Accumulator
			flag2seg 			: out STD_LOGIC_VECTOR (3 downto 0); -- Flags
			busOut2seg 			: out STD_LOGIC_VECTOR (7 downto 0); -- Value on the bus
			disp2seg			: out STD_LOGIC_VECTOR(7 downto 0); --Display register
			errSig2seg 			: out STD_LOGIC; -- Bus Error signal
			ovf 				: out STD_LOGIC; -- Overflow
			zero 				: out STD_LOGIC); -- Zero
end component;

------- Mapped processor signals
signal CLK 					: std_logic := '0';				
signal master_load_enable	: std_logic := '0';
signal ARESETN 			    : std_logic := '0';
signal pc2seg 				: std_logic_vector(7 downto 0);
signal instr2seg			: std_logic_vector(11 downto 0);
signal Addr2seg 			: std_logic_vector(7 downto 0);
signal dMemOut2seg			: std_logic_vector(7 downto 0);
signal aluOut2seg 			: std_logic_vector(7 downto 0);
signal acc2seg				: std_logic_vector(7 downto 0);
signal flag2seg 			: std_logic_vector(3 downto 0);
signal busOut2seg 			: std_logic_vector(7 downto 0);
signal disp2seg			    : std_logic_vector(7 downto 0);
signal errSig2seg 			: std_logic;
signal ovf 				    : std_logic;
signal zero 				: std_logic;

------------------- Help signals
signal time_step : integer := 0;

constant DATA_WIDTH : integer := 8;
type MEMORY_ARRAY is array(0 to (2**8)-1) of std_logic_vector(DATA_WIDTH-1 downto 0);

impure function init_memory_wfile(mif_file_name : in string) return MEMORY_ARRAY is
    file mif_file : text open read_mode is mif_file_name;
    variable mif_line : line;
    variable temp_bv : bit_vector(DATA_WIDTH-1 downto 0);
    variable temp_mem : MEMORY_ARRAY;
	variable i : integer := 0;
begin
    while not endfile(mif_file) loop
        readline(mif_file, mif_line);
			read(mif_line, temp_bv);
			temp_mem(i) := to_stdlogicvector(temp_bv);
			i := i + 1;
    end loop;
    return temp_mem;
end function;

signal disp2seg_expected 	: MEMORY_ARRAY := init_memory_wfile("disptrace.txt");
signal acc2seg_expected 	: MEMORY_ARRAY := init_memory_wfile("acctrace.txt");
signal pc2seg_expected 		: MEMORY_ARRAY := init_memory_wfile("pctrace.txt");
signal dMemOut2seg_expected : MEMORY_ARRAY := init_memory_wfile("dMemOuttrace.txt");
signal flag2seg_expected 	: MEMORY_ARRAY := init_memory_wfile("flagtrace.txt");

signal disp2seg_index 		: integer := 0;	
signal acc2seg_index 		: integer := 0;
signal pc2seg_index			: integer := 0;
signal dMemOut2seg_index 	: integer := 0;
signal flag2seg_index 		: integer := 0;

begin

	CLK <= not CLK after 5 ns; -- Clkperiod 10ns.

	process(CLK)
	begin
		if rising_edge(CLK) then 
			time_step <= time_step + 1;
			master_load_enable <= not master_load_enable;
		else
			master_load_enable <= not master_load_enable;
		end if;
		if time_step = 2 then
			ARESETN <= '1'; -- release reset
		end if;
	end process;
 
	disp2seg_check : process(disp2seg)
	begin
		if ARESETN = '1' then 
			if disp2seg_expected(disp2seg_index) = 144 then 
				assert false
					report "Mission Succcess, the galaxy is at peace"
					severity FAILURE;
			else
				assert disp2seg = disp2seg_expected(disp2seg_index)
					report "Unexpected value in <disp>!"
					severity FAILURE;
				disp2seg_index <= disp2seg_index + 1;
			end if;
		end if;
	end process;
	
	acc2seg_check : process(acc2seg)
	begin
		if ARESETN = '1' then 
			assert acc2seg = acc2seg_expected(acc2seg_index)
				report "Unexpected value in <acc>!"
				severity FAILURE;
			acc2seg_index <= acc2seg_index + 1;
		end if;
	end process;
		
	pc2seg_check : process(pc2seg)
	begin
		if ARESETN = '1' then 
			assert pc2seg = pc2seg_expected(pc2seg_index)
				report "Unexpected value in <pc>!"
				severity FAILURE;
			pc2seg_index <= pc2seg_index + 1;
		end if;
	end process;
		
	dMemOut2seg_check : process(dMemOut2seg)
	begin
		if ARESETN = '1' then 
			assert dMemOut2Seg = dMemOut2Seg_expected(dMemOut2Seg_index)
				report "Unexpected value in <dMemOut>!"
				severity FAILURE;
			dMemOut2Seg_index <= dMemOut2Seg_index + 1;
		end if;
	end process;
		
	flag2seg_check : process(flag2seg)
	begin
		if ARESETN = '1' then 
			assert flag2seg = flag2seg_expected(flag2seg_index)
				report "Unexpected value in <flag>!"
				severity FAILURE;
			flag2seg_index <= flag2seg_index + 1;
		end if;
	end process;
 

	processor_DUT : eda322_processor port map (
		"00000000",  --extin tied low.
		CLK 				,
		master_load_enable	,
		ARESETN 			,
		pc2seg 				,
		instr2seg			,
		Addr2seg 			,
		dMemOut2seg			,
		aluOut2seg 			,
		acc2seg				,
		flag2seg 			,
		busOut2seg 			,
		disp2seg			,
		errSig2seg 			,
		ovf 				,
		zero 				
		);
end architecture;
	
	