--****************************************************************************************************************--
--! @ CLA
--! @ Carry Lookahead Adder. A type of adder the computes the carry concurrent to the sum, resulting in a much shorter critical path for the carry signal.

--! Authors: 
--! John William Croft 
--****************************************************************************************************************--

----------------------------------------------------- Libraries ----------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;

----------------------------------------------------- Entity -------------------------------------------------------
entity CLA is
	-- number of bits.
	generic(N : integer := 8);

	port (	S : out std_logic_vector(N-1 downto 0);
			A : in std_logic_vector(N-1 downto 0);
			B : in std_logic_vector(N-1 downto 0);
			Cin : in std_logic;
			Cout : out std_logic
   );
end entity CLA;
----------------------------------------------------- Architecture ------------------------------------------------- 
architecture struct of CLA is
	signal g, p : 	std_logic_vector(N-1 downto 0);	-- generate & propagate signals
	signal c_internal 	:	std_logic_vector(N downto 0);  -- Note the vector length, the Nth bit is Cout and not used for computation.
	
begin
	c_internal(0) <= Cin;
	Cout <= c_internal(N) ;
	
	G1: for i in 0 to N-1 generate -- dataflow 'algebra' derived from Full Adder truth table (see lecture 2).
				-- FA (without Cout logic, as it is generated på CLA!)
		CLA :	s(i) <= ( c_internal(i) XOR ( A(i) XOR B(i) ) );
				-- g/p compute
				g(i) <= ( A(i) AND B(i) );
				p(i) <= ( A(i) XOR B(i) );
				-- Carry compute
				LAi: c_internal(i+1) <= (g(i) OR (p(i) AND c_internal(i)));
	end generate;
	
	
	---------- THIS IS THE ONE ----------
	---- FA (without Cout logic as it is generated på CLA!)
	--G1: for i in 0 to N-1 generate
	--	Ai : s(i) <= ( c_internal(i) XOR ( A(i) XOR B(i) ) );
	--	end generate;
	---- g/p compute
	--G2: for i in 0 to N-1 generate
	--	GPi : 	g(i) <= ( A(i) AND B(i) );
	--			p(i) <= ( A(i) XOR B(i) );
	--	end generate;
	---- Carry compute
	--CLA: for i in 0 to N-1 generate
	--	LAi: c_internal(i+1) <= (g(i) OR (p(i) AND c_internal(i)));
	--end generate;
		
		
	---- LA1
	--G_LA1: for i in 0 to N-1 generate
	--	COND1: if (i mod 2) = 0 generate
	--		LA1: 	c_internal(i+1) <= ((c_internal(0) AND p(i)) OR g(i));
	--				P_LA(i) <= (p(i) AND p(i+1));
	--				G_LA(i) <= ((g(i) AND p(i+1)) OR g(i+1));
	--end generate;
	---- LA2
	--G_LA2: for i in 0 to (N/2)-1 generate
	--	COND1: if (i mod 2) = 0 generate
	--		LA2: 	c_internal(i+1) <= ((c_internal(0) AND P_LA(i)) OR G_LA(i));
	--				P_LA(i) <= (P_LA(i) AND P_LA(i+1));
	--				G_LA(i) <= ((G_LA(i) AND P_LA(i+1)) OR G_LA(i+1));
end architecture struct;

